"""Access to Python's configuration information."""

import os
import sys
import os.path

__all__ = [
    'get_config_var',
    'get_config_vars',
    'get_path',
    'get_path_names',
    'get_paths',
    'get_platform',
    'get_python_version',
    'get_scheme_names',
]

_INSTALL_SCHEMES = {
    'bundled': {
        'stdlib': '{bundled_root}/stdlib.zip',
        'platstdlib': '{bundled_root}/modules',
        'purelib': '{bundled_root}/site-packages',
        'platlib': '{bundled_root}/site-packages',
        'data': '{bundled_root}/share',
        },
    }

_SCHEME_KEYS = ('stdlib', 'platstdlib', 'purelib', 'platlib', 'data')
_PY_VERSION_SHORT = f'{sys.version_info[0]}.{sys.version_info[1]}'
_CONFIG_VARS = None


def _subst_vars(s, local_vars):
    try:
        return s.format(**local_vars)
    except KeyError:
        try:
            return s.format(**os.environ)
        except KeyError as var:
            raise AttributeError('{%s}' % var)


def _extend_dict(target_dict, other_dict):
    target_keys = target_dict.keys()
    for key, value in other_dict.items():
        if key in target_keys:
            continue
        target_dict[key] = value


def _expand_vars(scheme, vars):
    res = {}
    if vars is None:
        vars = {}
    _extend_dict(vars, {'bundled_root': os.path.dirname(sys.executable)})

    for key, value in _INSTALL_SCHEMES[scheme].items():
        normalized_path = os.path.normpath(_subst_vars(value, vars))
        if os.path.exists(normalized_path):
            res[key] = normalized_path
    return res


def _get_default_scheme():
    return 'bundled'


def _init_config_vars(vars):
    from _sysconfigdata import build_time_vars
    vars.update(build_time_vars)

#
# public APIs
#

def get_scheme_names():
    """Return a tuple containing the schemes names."""
    return tuple(sorted(_INSTALL_SCHEMES))


def get_path_names():
    """Return a tuple containing the paths names."""
    return _SCHEME_KEYS


def get_paths(scheme=_get_default_scheme(), vars=None, expand=True):
    """Return a mapping containing an install scheme.

    ``scheme`` is the install scheme name. If not provided, it will
    return the default scheme for the current platform.
    """
    if expand:
        return _expand_vars(scheme, vars)
    else:
        return _INSTALL_SCHEMES[scheme]


def get_path(name, scheme=_get_default_scheme(), vars=None, expand=True):
    """Return a path corresponding to the scheme.

    ``scheme`` is the install scheme name.
    """
    return get_paths(scheme, vars, expand)[name]


def get_config_vars(*args):
    """With no arguments, return a dictionary of all configuration
    variables relevant for the current platform.

    With arguments, return a list of values that result from looking up
    each argument in the configuration variable dictionary.
    """
    global _CONFIG_VARS
    if _CONFIG_VARS is None:
        _CONFIG_VARS = {}
        _init_config_vars(_CONFIG_VARS)

    if args:
        vals = []
        for name in args:
            vals.append(_CONFIG_VARS.get(name))
        return vals
    else:
        return _CONFIG_VARS


def get_config_var(name):
    """Return the value of a single variable using the dictionary returned by
    'get_config_vars()'.

    Equivalent to get_config_vars().get(name)
    """
    return get_config_vars().get(name)


def get_platform():
    """Return a string that identifies the current platform."""
    if os.name == 'nt':
        # sniff sys.version for architecture
        prefix = " bit ("
        i = sys.version.find(prefix)
        if i == -1:
            return sys.platform
        j = sys.version.find(")", i)
        look = sys.version[i+len(prefix):j].lower()
        if look == 'amd64':
            return 'win-amd64'
        if look == 'itanium':
            return 'win-ia64'
        return sys.platform

    osname, host, release, version, machine = os.uname()

    osname = osname.lower().replace('/', '')
    machine = machine.replace(' ', '_')
    machine = machine.replace('/', '-')

    if osname[:6] == "darwin":
        osname = 'macosx'
        machine = 'i386'
        if sys.maxsize >= 2**32:
            machine = 'x86_64'

    return "%s-%s" % (osname, machine)


def get_python_version():
    return _PY_VERSION_SHORT


def _print_dict(title, data):
    for index, (key, value) in enumerate(sorted(data.items())):
        if index == 0:
            print('%s: ' % (title))
        print('\t%s = "%s"' % (key, value))


def _main():
    print('Platform: "%s"' % get_platform())
    print('Python version: "%s"' % get_python_version())
    print('Current installation scheme: "%s"' % _get_default_scheme())
    _print_dict('Paths', get_paths())
    _print_dict('Variables', get_config_vars())


if __name__ == '__main__':
    _main()
