import gzip
import os
import platform
import logging
from datetime import datetime
for handler in logging.root.handlers[:]:
    logging.root.removeHandler(handler)
from logging.handlers import TimedRotatingFileHandler
import shutil

system = platform.system()
base_path = os.environ.get('SystemDrive', 'C:') + '\\'
acrodir = "Acronis"
backuper_directory = "BackupAndRecovery"
mms_folder = "MMS"
file_name = "staging.log"

def find_directory_path_matching_sequence(start_dir, sequence):
    for root, dirs, files in os.walk(start_dir):
        parts = os.path.normpath(root).split(os.sep)
        if len(parts) >= len(sequence):
            for i in range(len(parts) - len(sequence) + 1):
                if parts[i:i+len(sequence)] == sequence:
                    return root
    return None

def resolve_long_path(path):
    try:
        import ctypes
        buf = ctypes.create_unicode_buffer(4096)
        get_long_path = ctypes.windll.kernel32.GetLongPathNameW
        r = get_long_path(path, buf, 4096)
        if r > 0 and r <= 4096:
            return buf.value
    except Exception:
        pass
    return path

def get_log_file_path():
    try:
        system = platform.system()
        
        if system == 'Windows':
                try:
                    release = int(platform.release())

                    if release >= 6:
                        programdata_path = os.path.expandvars('%PROGRAMDATA%')

                        if programdata_path and os.path.exists(programdata_path):
                            return programdata_path
                        else:
                            print("[get_log_file_path] PROGRAMDATA not found or not accessible.")

                except ValueError:
                    # platform.release() was not numeric:
                    base_path = os.environ.get('SystemDrive', 'C:') + '\\'
                    appdata = find_directory_path_matching_sequence(base_path, [acrodir, backuper_directory, mms_folder])

                    if appdata and os.path.exists(appdata):
                        root_log_dir = os.path.normpath(appdata)
                        for _ in range(len([acrodir, backuper_directory, mms_folder])):
                            root_log_dir = os.path.dirname(root_log_dir)
                        return root_log_dir
                    else:
                        print("[get_log_file_path] No Acronis path found.")

                # fallback for both Vista failure and XP failure
                temp_path = os.path.expandvars('%TEMP%')
                temp_path_long = resolve_long_path(temp_path)

                if temp_path_long and os.path.exists(temp_path_long):
                    return temp_path_long
                else:
                    print("[get_log_file_path] No usable TEMP path found.")
                    return

        elif system == 'Linux':
            if os.path.exists("/var/lib"):
                return "/var/lib"
            else:
                print("[get_log_file_path] /var/lib does not exist.")
                return

        elif system == 'Darwin':  # macOS
            if os.path.exists("/Library/Application Support"):
                return "/Library/Application Support"
            else:
                print("[get_log_file_path] /Library/Application Support does not exist.")
                return

        else:
            print("[get_log_file_path] OS not recognized.")
            return

    except Exception as e:
        print("[get_log_file_path] Error: {}".format(e))
        return
    
def log_rotator(source, dest):
    if not dest.endswith(".gz"):
        with open(source, "rb") as f_in, gzip.open(dest + ".gz", "wb") as f_out:
            shutil.copyfileobj(f_in, f_out)
        os.remove(source) # Remove uncompressed log

def setup_logger():
    root_log_directory = get_log_file_path()
    if not root_log_directory:
        print("[setup_logger] No valid log directory. Exiting without logging.")
        return
    full_log_dir = os.path.join(root_log_directory, acrodir, backuper_directory, mms_folder)
    
    if not os.path.exists(full_log_dir):
        try:
            os.makedirs(full_log_dir)
        except OSError as e:
            print("[setup_logger] Failed to create log directory in Temp:", full_log_dir)
            prnt("[setup_logger] Error: {}".format(e))
            return

    log_file = os.path.join(full_log_dir, file_name)

    try:
        file_handler = TimedRotatingFileHandler(log_file, when='midnight', backupCount=30, encoding="utf-8")
        file_handler.suffix = "%Y-%m-%d"
        file_handler.rotator = log_rotator
        
        formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
        file_handler.setFormatter(formatter)

        logging.basicConfig(level=logging.INFO, handlers=[file_handler])
        logging.info('-------STAGING LOG STARTED--------')

    except Exception as e:
        # Graceful exit if file handler setup fails
        print("[setup_logger] Failed to set up file handler for logging. Error: {}".format(e))
        return

setup_logger()
