import argparse
import acrort
import sys


class ArgumentsHelp:
    def __init__(self, description=None, epilog=None):
        self._description = description
        self._epilog = epilog

    @property
    def description(self):
        if self._description is None:
            return ''
        return self._description

    @property
    def epilog(self):
        if self._epilog is None:
            return ''
        return self._epilog


class ArgumentsProcessor:
    def __init__(self):
        pass

    def create_arguments_parser(self):
        return None

    def process_arguments(self, config, args):
        pass

    def get_help(self):
        return ArgumentsHelp()


class ArgumentParser(argparse.ArgumentParser):
    def __init__(*args, **kwargs):
        argparse.ArgumentParser.__init__(*args, **kwargs)

    def exit(self, status=0, message=None):
        if message:
            self._print_message(message, sys.stderr)
        if status == 0:
            acrort.common.make_script_error(acrort.common.ON_HELP_RETURN_CODE, "argparse help is given").throw()
        acrort.common.make_script_error(acrort.common.INVALID_USE_RETURN_CODE, "argparse error").throw()


class CommandLineParser:
    def __init__(self):
        self._processors = []
        self._custom_parser = ArgumentParser(add_help=False)

    def add_argument(self, *args, **kwargs):
        if self._custom_parser:
            self._custom_parser.add_argument(*args, **kwargs)

    def parse_arguments(self, *input):
        description = ''
        epilog = ''
        parent_parsers = []
        for processor in self._processors:
            help = processor.get_help()
            description += help.description
            epilog += help.epilog
            parser = processor.create_arguments_parser()
            if parser is not None:
                parent_parsers.append(parser)
        if self._custom_parser:
            parent_parsers.append(self._custom_parser)

        parser = ArgumentParser(description=description, epilog=epilog, formatter_class=argparse.RawDescriptionHelpFormatter, parents=parent_parsers)
        args = parser.parse_args(*input)
        config = {}
        config['args'] = args
        for processor in self._processors:
            processor.process_arguments(config, args)
        return config

    def append_processor(self, processor):
        self._processors.append(processor)
